<?php

namespace App\Http\Controllers;

use App\Models\ArchingCash;
use App\Models\Billing;
use App\Models\Business;
use App\Models\Client;
use App\Models\Currency;
use App\Models\Destination;
use App\Models\DetailBilling;
use App\Models\DetailPayment;
use App\Models\DetailReferenceGuide;
use App\Models\DetailSaleNote;
use App\Models\Driver;
use App\Models\IdentityDocumentType;
use App\Models\IgvTypeAffection;
use App\Models\Origin;
use App\Models\PayMode;
use App\Models\PayModeGuide;
use App\Models\Product;
use App\Models\ReferenceGuide;
use App\Models\SaleNote;
use App\Models\Sender;
use App\Models\Serie;
use App\Models\StatusGuide;
use App\Models\TypeDocument;
use App\Models\Unit;
use App\Models\User;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade as PDF;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Luecano\NumeroALetras\NumeroALetras;
use Illuminate\Support\Facades\Auth;

class ReferenceGuideController extends Controller
{
    public function index()
    {
        $data['modo_pagos_save']        = PayMode::get();
        $data['modo_pagos']             = PayModeGuide::get();
        $data['drivers']                = Driver::get();
        $data['status_guides']          = StatusGuide::get();
        $data['type_documents']         = IdentityDocumentType::where('estado', 1)->get();
        return view('admin.reference_guides.list', $data);
    }

    public function get()
    {
        $reference_guides    = ReferenceGuide::select('reference_guides.*', 'clients.nombres as nombre_cliente',
                        'clients.dni_ruc',
                            'districts.descripcion as origen_provincia',
                            'origins.direccion as origen_direccion',
                            'disc_d.descripcion as destino_provincia',
                            'destinations.direccion as destino_direccion',
                            'pay_mode_guides.descripcion as tipo_pago')
                            ->join('clients', 'reference_guides.iddestinatario', 'clients.id')
                            ->join('origins', 'reference_guides.idorigen', 'origins.id')
                            ->join('districts', 'origins.iddistrito', 'districts.id')
                            ->join('destinations', 'reference_guides.iddestino', 'destinations.id')
                            ->join('districts as disc_d', 'destinations.iddistrito', 'disc_d.id')
                            ->join('pay_mode_guides', 'reference_guides.idforma_pago', 'pay_mode_guides.id')
                            ->orderBy('id', 'DESC')
                            ->get();
        return Datatables()
                    ->of($reference_guides)
                    ->addColumn('destinatario', function ($reference_guides) {
                        $cliente  = $reference_guides->nombre_cliente;
                        return $cliente;
                    })
                    ->addColumn('documento', function ($reference_guides) {
                        $documento  = $reference_guides->serie . '-' . $reference_guides->correlativo;
                        return $documento;
                    })
                    ->addColumn('fecha_traslado', function ($reference_guides) {
                        $fecha_traslado = date('d-m-Y', strtotime($reference_guides->fecha_traslado));
                        return $fecha_traslado;
                    })
                    ->addColumn('estado', function ($reference_guides) 
                    {
                        $estado    = $reference_guides->estado;
                        $btn    = '';
                        switch ($estado) {
                            case '1':
                                $btn .= '<span class="badge text-white" style="background-color: rgb(108, 117, 125);">Registrado</span>';
                                break;

                            case '2':
                                $btn .= '<span class="badge bg-warning text-white">En tránsito</span>';
                                break;

                            case '3':
                                $btn .= '<span class="badge bg-info text-white">En destino</span>';
                                break;

                            case '4':
                                $btn .= '<span class="badge bg-success text-white">Entregado</span>';
                                break;
                        }
                        return $btn;
                    })
                    ->addColumn('acciones', function($reference_guides){
                        /* <a class="dropdown-item btn-open-whatsapp" data-id="' . $id . '" href="javascript:void(0);">
                                                <i class="fa-regular fa-envelope"></i>
                                                <span> Enviar Documento</span>
                        </a> */
                        $id     = $reference_guides->id;
                        $btn    = '<div class="dropdown">
                                        <button type="button" class="btn p-0 dropdown-toggle hide-arrow" data-bs-toggle="dropdown"><svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24"><path fill="none" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12h18M3 6h18M3 18h18"/></svg></button>
                                        <div class="dropdown-menu">
                                            <a class="dropdown-item btn-view" data-id="'.$id.'" href="javascript:void(0);">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-eye"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path><circle cx="12" cy="12" r="3"></circle></svg>
                                                <span>Ver detalle</span>
                                            </a>
                                            <a class="dropdown-item btn-detail" data-id="'.$id.'" href="javascript:void(0);">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-edit-2"><path d="M17 3a2.828 2.828 0 1 1 4 4L7.5 20.5 2 22l1.5-5.5L17 3z"></path></svg>
                                                <span> Actualizar Información</span>
                                            </a>
                                            <a class="dropdown-item btn-gen-voucher" data-id="'.$id.'" href="javascript:void(0);">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-file-text"><path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path><polyline points="14 2 14 8 20 8"></polyline><line x1="16" y1="13" x2="8" y2="13"></line><line x1="16" y1="17" x2="8" y2="17"></line><polyline points="10 9 9 9 8 9"></polyline></svg>
                                                <span>Generar Boleta/Factura</span>
                                            </a>
                                            <a class="dropdown-item btn-gen-sale-note" data-id="'.$id.'" href="javascript:void(0);">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-file"><path d="M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"></path><polyline points="13 2 13 9 20 9"></polyline></svg>
                                                <span>Generar Nota de Venta</span>
                                            </a>
                                            <a class="dropdown-item btn-print" data-id="' . $id . '" href="javascript:void(0);">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-printer"><polyline points="6 9 6 2 18 2 18 9"></polyline><path d="M6 18H4a2 2 0 0 1-2-2v-5a2 2 0 0 1 2-2h16a2 2 0 0 1 2 2v5a2 2 0 0 1-2 2h-2"></path><rect x="6" y="14" width="12" height="8"></rect></svg>
                                                <span> Imprimir Ticket</span>
                                            </a>
                                        </div>
                                    </div>';
                        return $btn;
                    })
                    ->rawColumns(['fecha_traslado','acciones', 'estado', 'destinatario'])
                    ->make(true);   
    }

    public function create()
    {
        $iduser             = Auth::user()['id'];
        $idcash             = Auth::user()['idcaja'];
        $search             = count(ArchingCash::where('idcaja', $idcash)->where('idusuario', $iduser)->where('estado', 1)->get());
        if ($search < 1) 
        {
            return redirect(route('admin.reference_guides'))->with('alert', 'Primero debe aperturar caja');
        }
        $data['type_documents_p']       = TypeDocument::where('id', 8)->get();
        $data['type_documents']         = IdentityDocumentType::where('estado', 1)->get();
        $data['clients']                = Client::orderBy('id', 'DESC')->get();
        $data['senders']                = Sender::get();
        $data['modo_pagos']             = PayModeGuide::get();
        $data["units"]                  = Unit::where('estado', 1)->get();
        $data['type_inafects']          = IgvTypeAffection::where('estado', 1)->get();
        $data['products']               = Product::get();
        $data['drivers']                = Driver::get();
        $data['origins']                = Origin::select('origins.*', 'districts.descripcion as distrito')
            ->join('districts', 'origins.iddistrito', 'districts.id')
            ->get();
        $data['destinations']           = Destination::select('destinations.*', 'districts.descripcion as distrito')
            ->join('districts', 'destinations.iddistrito', 'districts.id')
            ->get();

        return view('admin.reference_guides.create.home', $data);
    }

    public function load_serie(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'title'     => 'Espere',
                'type'      => 'warning'
            ]);
            return;
        }

        $idcaja     = Auth::user()['idcaja'];
        $serie      = Serie::where('id', '>', 4)->where('idtipo_documento', 8)->where('idcaja', $idcaja)->first();
        echo json_encode(['status'  => true, 'serie'  => $serie]);
    }

    public function get_up_senders(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $idtipo_documento = $request->input('idtipo_documento');
        if (empty($idtipo_documento)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Seleccione un tipo de comprobante',
                'type'      => 'warning'
            ]);
            return;
        }

        $type_document                      = TypeDocument::where('id', $idtipo_documento)->first();
        $serie                              = Serie::where('idtipo_documento', $type_document->id)->first();
        $senders                            = Sender::orderBy('id', 'DESC')->get();

        echo json_encode(['status'  => true, 'serie'  => $serie, 'senders' => $senders]);
    }

    public function store(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        } 

        $id                 = $request->input('id');
        $fecha_traslado     = $request->input('fecha_traslado');
        $modo_pago          = $request->input('modo_pago');
        $idconductor        = $request->input('idconductor');
        $estado             = $request->input('estado');
        ReferenceGuide::where('id', $id)->update([
            'fecha_traslado'    => $fecha_traslado,
            'idforma_pago'      => $modo_pago,
            'idconductor'       => $idconductor,
            'estado'            => $estado
        ]);

        echo json_encode([
            'status'            => true,
            'msg'               => 'Datos actualizados con éxito',
            'type'              => 'success'
        ]);
    }

    public function get_up_clients(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $idtipo_documento = $request->input('idtipo_documento');
        if (empty($idtipo_documento)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Seleccione un tipo de comprobante',
                'type'      => 'warning'
            ]);
            return;
        }

        $type_document                      = TypeDocument::where('id', $idtipo_documento)->first();
        $serie                              = Serie::where('idtipo_documento', $type_document->id)->first();
        $clients                            = Client::orderBy('id', 'DESC')->get();
        echo json_encode(['status'  => true, 'serie'  => $serie, 'clients' => $clients]);
    }

    public function get_up_drivers(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $idtipo_documento = $request->input('idtipo_documento');
        if (empty($idtipo_documento)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Seleccione un tipo de comprobante',
                'type'      => 'warning'
            ]);
            return;
        }

        $type_document                      = TypeDocument::where('id', $idtipo_documento)->first();
        $serie                              = Serie::where('idtipo_documento', $type_document->id)->first();
        $drivers                            = Driver::orderBy('id', 'DESC')->get();
        echo json_encode(['status'  => true, 'serie'  => $serie, 'drivers' => $drivers]);
    }

    public function load_cart(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $cart           = $this->create_cart();
        $html_cart      = '';
        $html_totales   = '';
        $contador       = 0;

        if (!empty($cart['products'])) {
            foreach ($cart['products'] as $i => $product) {
                $contador   = $contador + 1;
                $html_cart .= '<tr>
                                <td class="text-center">' . $contador . '</td>
                                <td>' . $product["descripcion"] . '</td>
                                <td class="text-center">' . $product["unidad"] . '</td>
                                <td class="text-right">
                                    <div class="input-group input-group-sm">
                                        <span class="input-group-text btn-down" style="cursor: pointer;" data-id="' . $product["id"] . '" data-cantidad="' . $product["cantidad"] . '" data-precio="' . $product["precio_venta"] . '"><i class="ti ti-minus me-sm-1"></i></span>
                                        <input type="text" data-id="' . $product["id"] . '" class="quantity-counter text-center form-control" value="' . $product["cantidad"] . '">
                                        <span class="input-group-text btn-up" style="cursor: pointer;" data-id="' . $product["id"] . '" data-cantidad="' . $product["cantidad"] . '" data-precio="' . $product["precio_venta"] . '"><i class="ti ti-plus me-sm-1"></i></span>
                                    </div>
                                </td>
                                <td class="text-center"><input type="text" class="form-control form-control-sm text-center input-update" value="' . number_format($product["precio_venta"], 2, ".", "") . '" data-cantidad="' . $product["cantidad"] . '" data-id="' . $product["id"] . '" name="precio"></td>
                                <td class="text-center">' . number_format(($product["precio_venta"] * $product["cantidad"]), 2, ".", "") . '</td>
                                <td class="text-center"><span data-id="' . $product["id"] . '" class="text-danger btn-delete-product" style="cursor: pointer;"><svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-x align-middle mr-25"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg></span></td>
                            </tr>';
            }
        }

        $html_totales   .= '<div class="d-flex justify-content-between mb-2">
                                    <span class="w-px-100">OP. Gravadas:</span>
                                    <span class="fw-medium">S/' . number_format(($cart['exonerada'] + $cart['gravada'] + $cart['inafecta']), 2, ".", "") . '</span>
                                </div>
                                <div class="d-flex justify-content-between mb-2">
                                    <span class="w-px-100">IGV:</span>
                                    <span class="fw-medium">S/' . number_format($cart['igv'], 2, ".", "") . '</span>
                                </div>
                                <hr>
                                <div class="d-flex justify-content-between">
                                    <span class="w-px-100">Total:</span>
                                    <span class="fw-medium">S/' . number_format($cart['total'], 2, ".", "") . '</span>
                            </div>';

        echo json_encode([
            'status'        => true,
            'cart_products' => $cart,
            'html_cart'     => $html_cart,
            'html_totales'  => $html_totales
        ]);
    }

    public function add_product(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $id             = (int) $request->input('id');
        $cantidad       = (int) $request->input('cantidad');
        $precio         = number_format($request->input('precio'), 2, ".", "");

        if (!$this->add_product_cart($id, $cantidad, $precio)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Stock insuficiente',
                'type'      => 'warning'
            ]);
            return;
        }

        echo json_encode([
            'status'    => true,
            'msg'       => 'Producto agregado correctamente',
            'type'      => 'success'
        ]);
    }

    public function delete_product(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $id     = (int) $request->input('id');
        if (!$this->delete_product_cart($id)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'No se pudo eliminar el producto',
                'type'      => 'warning'
            ]);
            return;
        }

        echo json_encode([
            'status'    => true,
            'msg'       => 'Producto agregado correctamente',
            'type'      => 'success'
        ]);
    }

    public function store_product(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $id             = (int) $request->input('id');
        $cantidad       = (int) $request->input('cantidad');
        $precio         = number_format($request->input('precio'), 2, ".", "");
        if (!$this->update_quantity($id, $cantidad, $precio)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Stock insuficiente',
                'type'      => 'warning'
            ]);
            return;
        }

        echo json_encode([
            'status'    => true,
            'msg'       => 'Actualizado correctamente',
            'type'      => 'success'
        ]);
    }

    public function save(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $contrasena         = trim($request->input('contrasena'));
        if (empty($contrasena)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Ingrese una contraseña válida',
                'type'      => 'warning'
            ]);
            return;
        }

        if (!is_numeric($contrasena)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'La contraseña debe contener solo numeros',
                'type'      => 'warning'
            ]);
            return;
        }

        $serie              = $request->input('serie');
        $correlativo        = $request->input('correlativo');
        $fecha_traslado     = $request->input('fecha_traslado');
        $idorigen           = $request->input('idorigen');
        $iddestino          = $request->input('iddestino');
        $modo_pago          = $request->input('modo_pago');
        $idremitente        = $request->input('idremitente');
        $iddestinatario     = $request->input('iddestinatario');
        $idconductor        = $request->input('idconductor');
        $code               = $this->code__random();
        $search_code        = ReferenceGuide::where('codigo', $code)->first();

        if (!empty($search_code)) {
            $code           = $this->code__random();
        }

        if (empty($iddestino)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Seleccione un destino válido',
                'type'      => 'warning'
            ]);
            return;
        }

        if (empty($idremitente)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Seleccione un remitente',
                'type'      => 'warning'
            ]);
            return;
        }

        if (empty($idremitente)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Seleccione un destinatario',
                'type'      => 'warning'
            ]);
            return;
        }

        $cart               = $this->create_cart();
        $idcaja             = ArchingCash::where('idcaja', Auth::user()['idcaja'])->where('idusuario', Auth::user()['id'])->latest('id')->first()['id'];
        $business           = Business::where('id', 1)->first();
        $name_qr            = $serie . '-' . $correlativo;
        $registro           = $business->ruc . '|' . $serie . '|' . $correlativo;

        if(empty($cart["products"])) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Debe ingresar al menos 1 producto',
                'type'      => 'warning'
            ]);
            return;
        }

        // Gen qr
        QrCode::format('png')
            ->size(140)
            ->generate($registro, 'files/qrs/' . $name_qr . '.png');

        ReferenceGuide::insert([
            'serie'             => $serie,
            'correlativo'       => $correlativo,
            'codigo'            => $code,
            'fecha_emision'     => date('Y-m-d'),
            'hora'              => date('H:i:s'),
            'fecha_traslado'    => $fecha_traslado,
            'idorigen'          => $idorigen,
            'iddestino'         => $iddestino,
            'idremitente'       => $idremitente,
            'iddestinatario'    => $iddestinatario,
            'idconductor'       => $idconductor,
            'idforma_pago'      => $modo_pago,
            'exonerada'         => $cart["exonerada"],
            'inafecta'          => $cart["inafecta"],
            'gravada'           => $cart["gravada"],
            'exonerada'         => $cart["exonerada"],
            'anticipo'          => "0.00",
            'igv'               => $cart["igv"],
            'gratuita'          => "0.00",
            'otros_cargos'      => "0.00",
            'total'             => $cart["total"],
            'idusuario'         => Auth::user()['id'],
            'idcaja'            => $idcaja,
            'estado'            => 1,
            'password'          => $contrasena,
            'qr'                => $name_qr . '.png'
        ]);

        $latest_id              = ReferenceGuide::latest('id')->first()["id"];
        foreach ($cart["products"] as $product) {
            DetailReferenceGuide::insert([
                'idguia_referencia'     => $latest_id,
                'idproducto'            => $product['id'],
                'cantidad'              => $product['cantidad'],
                'descuento'             => 0.0000000000,
                'igv'                   => $product["igv"],
                'id_afectacion_igv'     => $product['idcodigo_igv'],
                'precio_unitario'       => $product['precio_venta'],
                'precio_total'          => ($product['precio_venta'] * $product['cantidad'])
            ]);
        }

        // Update serie
        $ultima_serie                   = Serie::where('idcaja', Auth::user()['idcaja'])->where('idtipo_documento', 8)->first();
        $ultimo_correlativo             = (int) $ultima_serie->correlativo + 1;
        $nuevo_correlativo              = str_pad($ultimo_correlativo, 8, '0', STR_PAD_LEFT);
        Serie::where('idcaja', Auth::user()['idcaja'])->where('idtipo_documento', 8)->update([
            'correlativo'   => $nuevo_correlativo
        ]);

        $data["business"]           = Business::where('id', 1)->first();
        $data["ubigeo"]             = $this->get_ubigeo($data["business"]->ubigeo);
        $data["tipo_comprobante"]   = TypeDocument::where('id', 8)->first();
        $data["reference_guide"]    = ReferenceGuide::select(
            'reference_guides.*',
            'districts.descripcion as origen_provincia',
            'origins.direccion as origen_direccion',
            'disc_d.descripcion as destino_provincia',
            'destinations.direccion as destino_direccion',
            'pay_mode_guides.descripcion as modo_pago_guia'
        )
            ->join('origins', 'reference_guides.idorigen', 'origins.id')
            ->join('districts', 'origins.iddistrito', 'districts.id')
            ->join('destinations', 'reference_guides.iddestino', 'destinations.id')
            ->join('districts as disc_d', 'destinations.iddistrito', 'disc_d.id')
            ->join('pay_mode_guides', 'reference_guides.idforma_pago', 'pay_mode_guides.id')
            ->where('reference_guides.id', $latest_id)
            ->first();


        $data["detalle"]            = DetailReferenceGuide::select('detail_reference_guides.*', 'products.descripcion as producto')
            ->join('products', 'detail_reference_guides.idproducto', 'products.id')
            ->where('detail_reference_guides.idguia_referencia', $latest_id)
            ->get();

        $formatter                  = new NumeroALetras();
        $data['numero_letras']      = $formatter->toWords($data["reference_guide"]->total, 2);

        $data["remitente"]          = Sender::where('id', $data["reference_guide"]->idremitente)->first();
        $data["destinatario"]       = Client::where('id', $data["reference_guide"]->iddestinatario)->first();
        $data["name"]               = $data['reference_guide']->serie . '-' . $data['reference_guide']->correlativo;
       
        // Acá está el problema!!! no se porque 
        $this->save_ticket_reference($data["name"] . '.pdf', $data);
        $this->destroy_cart();
        echo json_encode([
            'status'        => true,
            'idreference'   => $latest_id,
            'serie'         => $serie,
            'correlativo'   => $correlativo,
            'pdf'           => $data["name"] . '.pdf'
        ]);
    }

    public function save_ticket_reference($name, $data)
    {
        $customPaper        = array(0, 0, 520.00, 210.00);
        $pdf                = PDF::loadView('admin.reference_guides.create.ticket', $data)->setPaper($customPaper, 'landscape');
        return $pdf->save(public_path('files/reference-guides/' . $name));
    }

    public function print(Request $request) 
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }
        $id                         = $request->input('id');
        $customPaper                = array(0, 0, 630.00, 210.00);
        $data["business"]           = Business::where('id', 1)->first();
        $data["ubigeo"]             = $this->get_ubigeo($data["business"]->ubigeo);
        $data["tipo_comprobante"]   = TypeDocument::where('id', 8)->first();
        $data["reference_guide"]    = ReferenceGuide::select(
            'reference_guides.*',
            'districts.descripcion as origen_provincia',
            'origins.direccion as origen_direccion',
            'disc_d.descripcion as destino_provincia',
            'destinations.direccion as destino_direccion',
            'pay_mode_guides.descripcion as modo_pago_guia'
        )
            ->join('origins', 'reference_guides.idorigen', 'origins.id')
            ->join('districts', 'origins.iddistrito', 'districts.id')
            ->join('destinations', 'reference_guides.iddestino', 'destinations.id')
            ->join('districts as disc_d', 'destinations.iddistrito', 'disc_d.id')
            ->join('pay_mode_guides', 'reference_guides.idforma_pago', 'pay_mode_guides.id')
            ->where('reference_guides.id', $id)
            ->first();

        $data["name"]               = $data['reference_guide']->serie . '-' . $data['reference_guide']->correlativo;
        if(!file_exists(public_path('files/reference-guides/' . $data["name"] . '.pdf')))
        {
            $data["detalle"]            = DetailReferenceGuide::select('detail_reference_guides.*', 'products.descripcion as producto')
            ->join('products', 'detail_reference_guides.idproducto', 'products.id')
            ->where('detail_reference_guides.idguia_referencia', $id)
            ->get();

            $formatter                  = new NumeroALetras();
            $data['numero_letras']      = $formatter->toWords($data["reference_guide"]->total, 2);
            $data["remitente"]          = Sender::where('id', $data["reference_guide"]->idremitente)->first();
            $data["destinatario"]       = Client::where('id', $data["reference_guide"]->iddestinatario)->first();
            $pdf                = PDF::loadView('admin.reference_guides.create.ticket', $data)->setPaper($customPaper, 'landscape');
            $pdf->save(public_path('files/reference-guides/' . $data["name"] . '.pdf'));
        }

        echo json_encode([
            'status'    => true,
            'pdf'       => $data["name"] . '.pdf'
        ]);
    }

    public function view_detail(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }
        $id                         = $request->input('id');
        $data["reference_guide"]    = ReferenceGuide::select(
            'reference_guides.*',
            'districts.descripcion as origen_provincia',
            'origins.direccion as origen_direccion',
            'disc_d.descripcion as destino_provincia',
            'destinations.direccion as destino_direccion',
            'pay_mode_guides.descripcion as modo_pago_guia',
            'senders.nombres as remitente',
            'clients.nombres as destinatario'
        )
            ->join('origins', 'reference_guides.idorigen', 'origins.id')
            ->join('districts', 'origins.iddistrito', 'districts.id')
            ->join('destinations', 'reference_guides.iddestino', 'destinations.id')
            ->join('districts as disc_d', 'destinations.iddistrito', 'disc_d.id')
            ->join('pay_mode_guides', 'reference_guides.idforma_pago', 'pay_mode_guides.id')
            ->join('senders', 'reference_guides.idremitente', 'senders.id')
            ->join('clients', 'reference_guides.iddestinatario', 'clients.id')
            ->where('reference_guides.id', $id)
            ->first();

        $data["detalle"]            = DetailReferenceGuide::select('detail_reference_guides.*', 'products.descripcion as producto')
            ->join('products', 'detail_reference_guides.idproducto', 'products.id')
            ->where('detail_reference_guides.idguia_referencia', $id)
            ->get();

        echo json_encode([
            'status'            => true,
            'data'              => $data
        ]);
    }

    public function detail(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }
        $id                         = $request->input('id');
        $data["reference_guide"]    = ReferenceGuide::select(
            'reference_guides.*',
            'districts.descripcion as origen_provincia',
            'origins.direccion as origen_direccion',
            'disc_d.descripcion as destino_provincia',
            'destinations.direccion as destino_direccion',
            'pay_mode_guides.descripcion as modo_pago_guia',
            'senders.nombres as remitente',
            'clients.nombres as destinatario'
        )
            ->join('origins', 'reference_guides.idorigen', 'origins.id')
            ->join('districts', 'origins.iddistrito', 'districts.id')
            ->join('destinations', 'reference_guides.iddestino', 'destinations.id')
            ->join('districts as disc_d', 'destinations.iddistrito', 'disc_d.id')
            ->join('pay_mode_guides', 'reference_guides.idforma_pago', 'pay_mode_guides.id')
            ->join('senders', 'reference_guides.idremitente', 'senders.id')
            ->join('clients', 'reference_guides.iddestinatario', 'clients.id')
            ->where('reference_guides.id', $id)
            ->first();

        $data["detalle"]            = DetailReferenceGuide::select('detail_reference_guides.*', 'products.descripcion as producto')
            ->join('products', 'detail_reference_guides.idproducto', 'products.id')
            ->where('detail_reference_guides.idguia_referencia', $id)
            ->get();

        echo json_encode([
            'status'            => true,
            'data'              => $data
        ]);
    }

    public function gen_voucher(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $id                 = $request->input('id');
        $contrasena         = trim($request->input('contrasena'));
        $idpago             = $request->input('idpago');

        if(empty($contrasena)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Ingrese una contraseña válida',
                'type'      => 'warning'
            ]);
            return;
        }   

        $reference          = ReferenceGuide::where('id', $id)->first();
        if($reference->password != $contrasena) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'La contraseña ingresada es incorrecta',
                'type'      => 'warning'
            ]);
            return;
        }

        $detalle                = DetailReferenceGuide::select('detail_reference_guides.*', 'products.descripcion as producto',
                                'products.codigo_interno as codigo_interno','units.codigo as unidad', 'products.idcodigo_igv as idcodigo_igv',
                                'products.igv as igv', 'products.opcion as opcion')
                                ->join('products', 'detail_reference_guides.idproducto', '=', 'products.id')
                                ->join('units', 'products.idunidad', '=', 'units.id')
                                ->join('igv_type_affections', 'products.idcodigo_igv', 'igv_type_affections.id')
                                ->where('detail_reference_guides.idguia_referencia', $id)
                                ->get();

        $client                 = Client::where('id', $reference->iddestinatario)->first();
        $idtipo_comprobante     = (strlen($client->dni_ruc) == 11) ? 1 : 2;
        $fecha_emision          = date('Y-m-d');
        $fecha_vencimiento      = date('Y-m-d');
        $id_arching             = ArchingCash::where('idcaja', Auth::user()['idcaja'])->where('idusuario', Auth::user()['id'])->latest('id')->first()['id'];

        // Save
        $business                   = Business::where('id', 1)->first();
        $type_document              = TypeDocument::where('id', $idtipo_comprobante)->first();
        $client                     = Client::where('id', $client->id)->first();
        $identity_document          = IdentityDocumentType::where('id', $client->iddoc)->first();

        $ultima_serie               = Serie::where('idtipo_documento', $idtipo_comprobante)->where('idcaja', Auth::user()['idcaja'])->first();
        $ultimo_correlativo         = (int) $ultima_serie->correlativo;
        $serie                      = $ultima_serie->serie;
        $correlativo                = str_pad($ultimo_correlativo, 8, '0', STR_PAD_LEFT);
        
        $qr                         = $business->ruc . ' | ' . $type_document->codigo . ' | ' . $serie . ' | ' . $correlativo . ' | ' . number_format($reference->igv, 2, ".", "") . ' | ' . number_format($reference->total, 2, ".", "") . ' | ' . $fecha_emision . ' | ' . $identity_document->codigo . ' | ' . $client->dni_ruc;
        $name_qr                    = $serie . '-' . $correlativo;

        // Gen Qr
        QrCode::format('png')
        ->size(140)
        ->generate($qr, 'files/billings/qr/' . $name_qr . '.png');
        Billing::insert([
            'idtipo_comprobante'    => $idtipo_comprobante,
            'serie'                 => $serie,
            'correlativo'           => $correlativo,
            'fecha_emision'         => $fecha_emision,
            'fecha_vencimiento'     => $fecha_vencimiento,
            'hora'                  => date('H:i:s'),
            'idcliente'             => $client->id,
            'idmoneda'              => 1,
            'idpago'                => $idpago,
            'modo_pago'             => 1,
            'exonerada'             => $reference->exonerada,
            'inafecta'              => $reference->inafecta,
            'gravada'               => $reference->gravada,
            'anticipo'              => "0.00",
            'igv'                   => $reference->igv,
            'gratuita'              => "0.00",
            'otros_cargos'          => "0.00",
            'total'                 => $reference->total,
            'cdr'                   => 0,
            'anulado'               => 0,
            'id_tipo_nota_credito'  => null,
            'estado_cpe'            => 0,
            'errores'               => null,
            'nticket'               => null,
            'idusuario'             => Auth::user()['id'],
            'idcaja'                => $id_arching,
            'vuelto'                => "0.00",
            'qr'                    => $name_qr . '.png'
        ]);
        $idfactura                  = Billing::latest('id')->first()['id'];
        DetailPayment::insert([
            'idtipo_comprobante'    => $idtipo_comprobante,
            'idfactura'             => $idfactura,
            'idpago'                => $idpago,
            'monto'                 => $reference->total,
            'idcaja'                => $id_arching
        ]);

        foreach ($detalle as $product) {
            DetailBilling::insert([
                'idfacturacion'         => $idfactura,
                'idproducto'            => $product['idproducto'],
                'cantidad'              => $product['cantidad'],
                'descuento'             => 0.0000000000,
                'igv'                   => $product["igv"],
                'id_afectacion_igv'     => $product['idcodigo_igv'],
                'precio_unitario'       => $product['precio_unitario'],
                'precio_total'          => ($product['precio_unitario'] * $product['cantidad'])
            ]);

            if($product["opcion"] == 1)
            {
                if($product["stock"] != NULL) {
                    Product::where('id', $product["id"])->update([
                        "stock"  => $product["stock"] - $product["cantidad"]
                    ]);
                }
            }
        }

        $factura                = Billing::where('id', $idfactura)->first();
        $ruc                    = Business::where('id', 1)->first()->ruc;
        $code_sale              = TypeDocument::where('id', $factura->idtipo_comprobante)->first()->codigo;
        $name_sale              = $ruc . '-' . $code_sale . '-' . $factura->serie . '-' . $factura->correlativo;
        $id_sale                = $idfactura;
        $this->gen_ticket_b($idfactura, $name_sale);

        $ultima_serie_sale      = Serie::where('idtipo_documento', $idtipo_comprobante)->where('idcaja', Auth::user()['idcaja'])->first();
        $ultimo_correlativo_sale= (int) $ultima_serie_sale->correlativo + 1;
        $nuevo_correlativo_sale = str_pad($ultimo_correlativo_sale, 8, '0', STR_PAD_LEFT);
        Serie::where('idtipo_documento', $idtipo_comprobante)->where('idcaja', Auth::user()['idcaja'])->update([
            'correlativo'   => $nuevo_correlativo_sale
        ]);
        ReferenceGuide::where('id', $id)->update([
            'estado'        => 4
        ]);
        echo json_encode([
            'status'        => true,
            'id'            => $id_sale,
            'pdf'           => $name_sale . '.pdf',
            'type_document' => $idtipo_comprobante
        ]);
    }

    public function gen_voucher_sale_note(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $id                 = $request->input('id');
        $contrasena         = trim($request->input('contrasena'));
        $idpago             = $request->input('idpago');

        if(empty($contrasena)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Ingrese una contraseña válida',
                'type'      => 'warning'
            ]);
            return;
        }   

        $reference          = ReferenceGuide::where('id', $id)->first();
        if($reference->estado == 4) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'La encomienda ya fue entregada',
                'type'      => 'warning'
            ]);
            return;
        }

        if($reference->password != $contrasena) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'La contraseña ingresada es incorrecta',
                'type'      => 'warning'
            ]);
            return;
        }

        $detalle                = DetailReferenceGuide::select('detail_reference_guides.*', 'products.descripcion as producto',
                                'products.codigo_interno as codigo_interno','units.codigo as unidad', 'products.idcodigo_igv as idcodigo_igv',
                                'products.igv as igv', 'products.opcion as opcion')
                                ->join('products', 'detail_reference_guides.idproducto', '=', 'products.id')
                                ->join('units', 'products.idunidad', '=', 'units.id')
                                ->join('igv_type_affections', 'products.idcodigo_igv', 'igv_type_affections.id')
                                ->where('detail_reference_guides.idguia_referencia', $id)
                                ->get();

        $client                 = Client::where('id', $reference->iddestinatario)->first();
        $idtipo_comprobante     = 7;
        $fecha_emision          = date('Y-m-d');
        $fecha_vencimiento      = date('Y-m-d');
        $id_arching             = ArchingCash::where('idcaja', Auth::user()['idcaja'])->where('idusuario', Auth::user()['id'])->latest('id')->first()['id'];

        // Save
        $business                   = Business::where('id', 1)->first();
        $type_document              = TypeDocument::where('id', $idtipo_comprobante)->first();
        $identity_document          = IdentityDocumentType::where('id', $client->iddoc)->first();

        $ultima_serie               = Serie::where('idtipo_documento', $idtipo_comprobante)->where('idcaja', Auth::user()['idcaja'])->first();
        $ultimo_correlativo         = (int) $ultima_serie->correlativo;
        $serie                      = $ultima_serie->serie;
        $correlativo                = str_pad($ultimo_correlativo, 8, '0', STR_PAD_LEFT);

        SaleNote::insert([
            'idtipo_comprobante'    => $idtipo_comprobante,
            'serie'                 => $serie,
            'correlativo'           => $correlativo,
            'fecha_emision'         => $fecha_emision,
            'fecha_vencimiento'     => $fecha_vencimiento,
            'hora'                  => date('H:i:s'),
            'idcliente'             => $client->id,
            'idmoneda'              => 1,
            'idpago'                => $idpago,
            'modo_pago'             => 1,
            'exonerada'             => $reference->exonerada,
            'inafecta'              => $reference->inafecta,
            'gravada'               => $reference->gravada,
            'anticipo'              => "0.00",
            'igv'                   => $reference->igv,
            'gratuita'              => "0.00",
            'otros_cargos'          => "0.00",
            'total'                 => $reference->total,
            'estado'                => 1,
            'idusuario'             => Auth::user()['id'],
            'idcaja'                => $id_arching
        ]);

        $idnotaventa                = SaleNote::latest('id')->first()['id'];
        DetailPayment::insert([
            'idtipo_comprobante'    => $idtipo_comprobante,
            'idfactura'             => $idnotaventa,
            'idpago'                => $idpago,
            'monto'                 => $reference->total,
            'idcaja'                => $id_arching,
            'estado'                => 1
        ]);
        foreach ($detalle as $product) 
        {
            DetailSaleNote::insert([
                'idnotaventa'           => $idnotaventa,
                'idproducto'            => $product['idproducto'],
                'cantidad'              => $product['cantidad'],
                'descuento'             => 0.0000000000,
                'igv'                   => $product["igv"],
                'id_afectacion_igv'     => $product['idcodigo_igv'],
                'precio_unitario'       => $product['precio_unitario'],
                'precio_total'          => ($product['precio_unitario'] * $product['cantidad'])
            ]);

            if($product["stock"] != NULL)
            {
                Product::where('id', $product["id"])->update([
                    "stock"  => $product["stock"] - $product["cantidad"]
                ]);
            }
        }
        $ultima_serie                   = Serie::where('idtipo_documento', $idtipo_comprobante)->where('idcaja', Auth::user()['idcaja'])->first();
        $ultimo_correlativo             = (int) $ultima_serie->correlativo + 1;
        $nuevo_correlativo              = str_pad($ultimo_correlativo, 8, '0', STR_PAD_LEFT);
        Serie::where('idtipo_documento', $idtipo_comprobante)->where('idcaja', Auth::user()['idcaja'])->update([
            'correlativo'   => $nuevo_correlativo
        ]);

        // Gen ticket data to pdf
        $factura                        = SaleNote::where('id', $idnotaventa)->first();
        $ruc                            = Business::where('id', 1)->first()->ruc;
        $codigo_comprobante             = TypeDocument::where('id', $factura->idtipo_comprobante)->first()->codigo;
        $name                           = $ruc . '-' . $codigo_comprobante . '-' . $factura->serie . '-' . $factura->correlativo;
        $this->gen_ticket($idnotaventa, $name);

        ReferenceGuide::where('id', $id)->update([
            'estado'        => 4
        ]);
        echo json_encode([
            'status'    => true,
            'pdf'       => $name . '.pdf',
        ]);
    }

    public function gen_ticket_b($id, $name)
    {
        $customPaper                = array(0, 0, 630.00, 210.00);
        $data['business']           = Business::where('id', 1)->first();
        $data['ubigeo']             = $this->get_ubigeo($data['business']->ubigeo);
        $ruc                        = $data['business']->ruc;
        $factura                    = Billing::where('id', $id)->first();
        $codigo_comprobante         = TypeDocument::where('id', $factura->idtipo_comprobante)->first()->codigo;
        $data["name"]               = $ruc . '-' . $codigo_comprobante . '-' . $factura->serie . '-' . $factura->correlativo;

        $data['factura']            = Billing::where('id', $id)->first();
        $data['cliente']            = Client::where('id', $factura->idcliente)->first();
        $data['tipo_documento']     = IdentityDocumentType::where('id', $data['cliente']->iddoc)->first();
        $data['moneda']             = Currency::where('id', $factura->idmoneda)->first();
        $data['modo_pago']          = PayMode::where('id', $factura->modo_pago)->first();
        $data['detalle']            = DetailBilling::select(
            'detail_billings.*',
            'products.descripcion as producto',
            'products.codigo_interno as codigo_interno'
        )
            ->join('products', 'detail_billings.idproducto', '=', 'products.id')
            ->where('idfacturacion', $factura->id)
            ->get();

        $formatter                  = new NumeroALetras();
        $data['numero_letras']      = $formatter->toWords($factura->total, 2);
        $data['tipo_comprobante']   = TypeDocument::where('id', $factura->idtipo_comprobante)->first();
        $data['vendedor']           = mb_strtoupper(User::where('id', $data['factura']->idusuario)->first()->user);
        $data['payment_modes']      = DetailPayment::select('detail_payments.*', 'pay_modes.descripcion as modo_pago')
            ->join('pay_modes', 'detail_payments.idpago', 'pay_modes.id')
            ->where('idfactura', $factura->id)
            ->where('idtipo_comprobante', $factura->idtipo_comprobante)
            ->get();
        $data['count_payment']      = count($data['payment_modes']);
        $pdf                        = PDF::loadView('admin.billings.ticket_b', $data)->setPaper($customPaper, 'landscape');
        return $pdf->save(public_path('files/billings/ticket/' . $name . '.pdf'));
    }

    public function gen_ticket($id, $name)
    {
        $customPaper                = array(0, 0, 630.00, 210.00);
        $data['business']           = Business::where('id', 1)->first();
        $data['ubigeo']             = $this->get_ubigeo($data['business']->ubigeo);
        $ruc                        = $data['business']->ruc;
        $factura                    = SaleNote::where('id', $id)->first();
        $codigo_comprobante         = TypeDocument::where('id', $factura->idtipo_comprobante)->first()->codigo;
        $data["name"]               = $ruc . '-' . $codigo_comprobante . '-' . $factura->serie . '-' . $factura->correlativo;

        $data['factura']            = SaleNote::where('id', $id)->first();
        $data['cliente']            = Client::where('id', $factura->idcliente)->first();
        $data['tipo_documento']     = IdentityDocumentType::where('id', $data['cliente']->iddoc)->first();
        $data['moneda']             = Currency::where('id', $factura->idmoneda)->first();
        $data['modo_pago']          = PayMode::where('id', $factura->modo_pago)->first();
        $data['detalle']            = DetailSaleNote::select('detail_sale_notes.*', 'products.descripcion as producto', 
                                    'products.codigo_interno as codigo_interno')
                                    ->join('products', 'detail_sale_notes.idproducto', '=', 'products.id')
                                    ->where('idnotaventa', $factura->id)
                                    ->get();

        $formatter                  = new NumeroALetras();
        $data['numero_letras']      = $formatter->toWords($factura->total, 2);
        $data['tipo_comprobante']   = TypeDocument::where('id', $factura->idtipo_comprobante)->first();
        $data['vendedor']           = mb_strtoupper(User::where('id', $data['factura']->idusuario)->first()->user);
        $data['payment_modes']      = DetailPayment::select('detail_payments.*', 'pay_modes.descripcion as modo_pago')
            ->join('pay_modes', 'detail_payments.idpago', 'pay_modes.id')
            ->where('idfactura', $factura->id)
            ->where('idtipo_comprobante', $factura->idtipo_comprobante)
            ->get();
        $data['count_payment']      = count($data['payment_modes']);
        $pdf                        = PDF::loadView('admin.sale_notes.ticket', $data)->setPaper($customPaper, 'landscape');
        return $pdf->save(public_path('files/sale-notes/ticket/' . $name . '.pdf'));
    }

    public function search(Request $request) 
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $nro_orden          = trim($request->input('nro_orden'));
        $cod             = trim($request->input('codigo'));

        $search             = ReferenceGuide::where('correlativo', $nro_orden)->where('codigo', mb_strtoupper($cod))->first();
        if(empty($search)) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'No se encuentra el N° de Guía',
                'type'      => 'error'
            ]);
            return;
        }  
        
        $data["reference"]    = ReferenceGuide::select(
            'reference_guides.*',
            'districts.descripcion as origen_provincia',
            'origins.direccion as origen_direccion',
            'disc_d.descripcion as destino_provincia',
            'destinations.direccion as destino_direccion',
            'pay_mode_guides.descripcion as modo_pago_guia',
            'senders.nombres as remitente',
            'clients.nombres as destinatario'
        )
            ->join('origins', 'reference_guides.idorigen', 'origins.id')
            ->join('districts', 'origins.iddistrito', 'districts.id')
            ->join('destinations', 'reference_guides.iddestino', 'destinations.id')
            ->join('districts as disc_d', 'destinations.iddistrito', 'disc_d.id')
            ->join('pay_mode_guides', 'reference_guides.idforma_pago', 'pay_mode_guides.id')
            ->join('senders', 'reference_guides.idremitente', 'senders.id')
            ->join('clients', 'reference_guides.iddestinatario', 'clients.id')
            ->where('reference_guides.correlativo', $nro_orden)
            ->where('reference_guides.codigo', mb_strtoupper($cod))
            ->first();

        $data["detalle"]            = DetailReferenceGuide::select('detail_reference_guides.*', 'products.descripcion as producto')
            ->join('products', 'detail_reference_guides.idproducto', 'products.id')
            ->where('detail_reference_guides.idguia_referencia', $data["reference"]->id)
            ->get();

        echo json_encode([
            'status'    => true,
            'data'      => $data
        ]);
    }

    public function view_detail_confirm(Request $request) 
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $id                 = $request->input('id');
        $reference          = ReferenceGuide::where('id', $id)->first();
        $detalle            = DetailReferenceGuide::select('detail_reference_guides.*', 'products.descripcion as producto',
                                'products.codigo_interno as codigo_interno','units.codigo as unidad', 'products.idcodigo_igv as idcodigo_igv',
                                'products.igv as igv', 'products.opcion as opcion')
                                ->join('products', 'detail_reference_guides.idproducto', '=', 'products.id')
                                ->join('units', 'products.idunidad', '=', 'units.id')
                                ->join('igv_type_affections', 'products.idcodigo_igv', 'igv_type_affections.id')
                                ->where('detail_reference_guides.idguia_referencia', $id)
                                ->get();
        echo json_encode([
            'status'    => true,
            'detalle'   => $detalle,
            'reference' => $reference
        ]);
    }

    public function view_detail_confirm_sale_note(Request $request) 
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'type'      => 'warning'
            ]);
            return;
        }

        $id                 = $request->input('id');
        $reference          = ReferenceGuide::where('id', $id)->first();
        $detalle            = DetailReferenceGuide::select('detail_reference_guides.*', 'products.descripcion as producto',
                                'products.codigo_interno as codigo_interno','units.codigo as unidad', 'products.idcodigo_igv as idcodigo_igv',
                                'products.igv as igv', 'products.opcion as opcion')
                                ->join('products', 'detail_reference_guides.idproducto', '=', 'products.id')
                                ->join('units', 'products.idunidad', '=', 'units.id')
                                ->join('igv_type_affections', 'products.idcodigo_igv', 'igv_type_affections.id')
                                ->where('detail_reference_guides.idguia_referencia', $id)
                                ->get();
        echo json_encode([
            'status'    => true,
            'detalle'   => $detalle,
            'reference' => $reference
        ]);
    }

    ## Functions to cart
    public function create_cart()
    {
        if (!session()->get('sale_note') || empty(session()->get('sale_note')['products'])) {
            $sale_note =
                [
                    'sale_note' =>
                    [
                        'products'     => [],
                        'igv'          => 0,
                        'exonerada'    => 0,
                        'gravada'      => 0,
                        'inafecta'     => 0,
                        'subtotal'     => 0,
                        'total'        => 0
                    ]
                ];

            session($sale_note);
            return session()->get('sale_note');
        }

        $exonerada  = 0;
        $gravada    = 0;
        $inafecta   = 0;
        $subtotal   = 0;
        $total      = 0;
        $igv        = 0;

        foreach (session('sale_note')['products'] as $index => $product) {
            if ($product['impuesto'] == 1) {
                $igv        +=  number_format((((float) $product['precio_venta'] - (float) $product['precio_venta'] / 1.18) * (int) $product['cantidad']), 2, ".", "");
                $igv        = $this->redondeado($igv);
            }

            if ($product["codigo_igv"] == "10") {
                $gravada    += number_format((((float) $product['precio_venta'] / 1.18) * (int) $product['cantidad']), 2, ".", "");
                $gravada     = $this->redondeado($gravada);
            }

            if ($product["codigo_igv"] == "20") {
                $exonerada   += number_format(((float) $product['precio_venta'] * (int) $product['cantidad']), 2, ".", "");
                $exonerada   = $this->redondeado($exonerada);
            }

            if ($product["codigo_igv"] == "30") {
                $inafecta    += number_format(((float) $product['precio_venta'] * (int) $product['cantidad']), 2, ".", "");
                $inafecta     = str_replace(',', '', $inafecta);
                $inafecta     = $this->redondeado($inafecta);
            }

            $subtotal      = $exonerada + $gravada + $inafecta;
            session()->put('sale_note.products.' . $index, $product);
        }

        $total      = $subtotal + $igv;

        $sale_note =
            [
                'sale_note' =>
                [
                    'products'     => session('sale_note')['products'],
                    'igv'          => $igv,
                    'exonerada'    => $exonerada,
                    'gravada'      => $gravada,
                    'inafecta'     => $inafecta,
                    'subtotal'     => $subtotal,
                    'total'        => $total,
                ]
            ];

        session($sale_note);
        return session()->get('sale_note');
    }

    public function add_product_cart($id, $cantidad, $precio)
    {
        $product        = Product::select(
            'products.*',
            'units.codigo as unidad',
            'igv_type_affections.descripcion as tipo_afecto',
            'igv_type_affections.codigo as codigo_igv'
        )
            ->join('units', 'products.idunidad', '=', 'units.id')
            ->join('igv_type_affections', 'products.idcodigo_igv', 'igv_type_affections.id')
            ->where('products.id', $id)
            ->first();

        if (!$product)
            return false;

        if ($product->stock != NULL) {
            if ($product->stock < $cantidad) {
                return false;
            } elseif ($product->stock == 0) {
                return false;
            }
        }

        $new_product    =
            [
                'id'                => $product->id,
                'codigo_sunat'      => $product->codigo_sunat,
                'descripcion'       => $product->descripcion,
                'idunidad'          => $product->idunidad,
                'unidad'            => $product->unidad,
                'idcodigo_igv'      => $product->idcodigo_igv,
                'codigo_igv'        => $product->codigo_igv,
                'igv'               => $product->igv,
                'precio_compra'     => $product->precio_compra,
                'precio_venta'      => $precio,
                'impuesto'          => $product->impuesto,
                'stock'             => $product->stock,
                'cantidad'          => $cantidad
            ];

        if (empty(session()->get('sale_note')['products'])) {
            session()->push('sale_note.products', $new_product);
            return true;
        }

        foreach (session()->get('sale_note')['products'] as $index => $product) {
            if ($id == $product['id']) {
                if ($product["stock"] != NULL) {
                    if ($product["stock"] < ($product['cantidad'] + $cantidad)) {
                        return false;
                    }
                }
                $product['cantidad'] = $product['cantidad'] + $cantidad;
                session()->put('sale_note.products.' . $index, $product);
                return true;
            }
        }

        session()->push('sale_note.products', $new_product);
        return true;
    }

    public function delete_product_cart($id)
    {
        if (!session()->get('sale_note') || empty(session()->get('sale_note')['products'])) {
            return false;
        }

        foreach (session()->get('sale_note')['products'] as $index => $product) {
            if ($id == $product['id']) {
                session()->forget('sale_note.products.' . $index, $product);
                return true;
            }
        }
    }

    public function update_quantity($id, $cantidad, $precio)
    {
        if (empty(session()->get('sale_note')['products'])) {
            return false;
        }

        foreach (session()->get('sale_note')['products'] as $index => $product) {
            if ($id == $product['id']) {
                if ($product["stock"] != NULL) {
                    if ($product["stock"] < $cantidad) {
                        return false;
                    } elseif ($product["stock"] == 0) {
                        return false;
                    }
                }
                $product['cantidad']           =  $cantidad;
                $product['precio_venta']       =  $precio;
                session()->put('sale_note.products.' . $index, $product);
                return true;
            }
        }
    }

    public function get_price_product(Request $request)
    {
        if (!$request->ajax()) {
            echo json_encode([
                'status'    => false,
                'msg'       => 'Intente de nuevo',
                'title'     => 'Espere',
                'type'      => 'warning'
            ]);
            return;
        }

        $id         = $request->input('id');
        $product    = Product::where('id', $id)->first();
        echo json_encode([
            'status'    => true,
            'product'   => $product
        ]);
    }

    public function destroy_cart()
    {
        if (!session()->get('sale_note') || empty(session()->get('sale_note')['products'])) {
            return false;
        }

        session()->forget('sale_note');
        return true;
    }

    public function tracking()
    {
        return view('tracking');
    }
}
